import {
    Entity,
    EntityDamageCause,
    EquipmentSlot,
    ItemComponentTypes,
    system,
    world,
} from "@minecraft/server";
import { Helper } from "../utils/Helper.js";
const dimensions = [
    "overworld",
    "nether",
    "the_end"
]
export default class Hedgehog {
    static ENTITY_OR_ITEM_ID = "sf_nba:hedgehog";
    static usedHedgehogTick = {};
    static blockInteractions = new Set();
    constructor() {
        this.initEvents();
    }
    initEvents() {
        world.beforeEvents.itemUseOn.subscribe(this.handleUseOnBlock.bind(this));
        world.afterEvents.itemReleaseUse.subscribe(this.handleRelease.bind(this));
        world.afterEvents.projectileHitEntity.subscribe(this.handleEntityHit.bind(this));
        system.afterEvents.scriptEventReceive.subscribe(this.handleHit.bind(this));
    }
    static extractHedgehogProperties(itemId) {
        const variant = itemId.replace(`${Hedgehog.ENTITY_OR_ITEM_ID}_`, "")[0];
        const isBaby = itemId.indexOf("baby") !== -1;
        return { variant, isBaby };
    }
    static applyHedgehogProperties(hedgehog, variant, isBaby, dye, owner = null, ownerId) {
        const tameable = hedgehog.getComponent("minecraft:tameable");
        hedgehog.triggerEvent("sf_nba:spawn_variant_" + variant);
        if (isBaby) {
            hedgehog.triggerEvent("sf_nba:spawn_baby");
        }
        hedgehog.setDynamicProperty('sf_nba:owner_id', ownerId)
        tameable?.tame(owner)
        if (dye) {
            hedgehog.setProperty('sf_nba:dye', dye)
        }
    }
    handleUseOnBlock(eventData) {
        const stack = eventData.itemStack;
        if (!stack) return;
        const itemId = stack.typeId;
        if (itemId.endsWith("spawn_egg")) return;
        if (!itemId.startsWith(Hedgehog.ENTITY_OR_ITEM_ID)) return;
        const player = eventData.source;
        const blockLocation = eventData.block.location;
        const blockFace = eventData.blockFace;
        Hedgehog.blockInteractions.add(player.id);
        eventData.cancel = true;
        const spawnLocation = this.calculateSpawnLocation(blockLocation, blockFace);
        const { variant, isBaby } = Hedgehog.extractHedgehogProperties(itemId);
        const dye = stack.getDynamicProperty('sf_nba:dye')
        system.run(() => {
            const entity = player.dimension.spawnEntity(Hedgehog.ENTITY_OR_ITEM_ID, spawnLocation);
            const playerDirection = player.getViewDirection();
            entity.setRotation({ x: -playerDirection.x, y: -playerDirection.y });
            Helper.decreaseMainhandItemStack(player, 1, stack);
            Hedgehog.applyHedgehogProperties(entity, variant, isBaby, dye, player, player.id);
        });
    }
    calculateSpawnLocation(blockLocation, blockFace) {
        const spawnLocation = {
            x: blockLocation.x + 0.5,
            y: blockLocation.y + 0.5,
            z: blockLocation.z + 0.5
        };
        switch (blockFace) {
            case 'Up':
                spawnLocation.y += 0.5;
                break;
            case 'Down':
                spawnLocation.y -= 0.5;
                break;
            case 'North':
                spawnLocation.z -= 0.5;
                break;
            case 'South':
                spawnLocation.z += 0.5;
                break;
            case 'West':
                spawnLocation.x -= 0.5;
                break;
            case 'East':
                spawnLocation.x += 0.5;
                break;
        }
        return spawnLocation;
    }
    handleRelease(eventData) {
        const stack = eventData.itemStack;
        const dye = stack?.getDynamicProperty('sf_nba:dye')
        if (!stack) return;
        const player = eventData.source;
        if (!player) return;
        if (!stack.type || !stack.type.id) return;
        const itemId = stack.type.id;
        if (!itemId.startsWith(Hedgehog.ENTITY_OR_ITEM_ID)) return;
        if (Hedgehog.usedHedgehogTick[player.id] === system.currentTick) return;
        if (Hedgehog.blockInteractions.has(player.id)) {
            Hedgehog.blockInteractions.delete(player.id);
            return;
        }
        const strength = 2;
        try {
            const cooldownComponent = stack.getComponent(ItemComponentTypes.Cooldown);
            if (cooldownComponent && cooldownComponent.getCooldownTicksRemaining(player) > 0) return;
            Hedgehog.usedHedgehogTick[player.id] = system.currentTick;
            const { variant, isBaby } = Hedgehog.extractHedgehogProperties(itemId);
            Hedgehog.spawnAndThrow(player, strength, variant, isBaby, dye);
            if (cooldownComponent) {
                cooldownComponent.startCooldown(player);
            }
            Helper.decreaseMainhandItemStack(player, 1, stack);
        } catch (error) {
            console.warn("Error in handleRelease:", error);
        }
    }
    handleHit(eventData) {
        const id = eventData.id;
        const entity = eventData.sourceEntity;
        if (!entity) return;
        const identifier = entity.typeId;
        if (identifier !== Hedgehog.ENTITY_OR_ITEM_ID || id !== "sf_nba:hedgehog_hit") return;
        system.runTimeout(() => {
            entity.triggerEvent("sf_nba:stop_rolling");
            entity.setProperty("sf_nba:hit", false);
        }, 2.2);
    }
    handleEntityHit(eventData) {
        const player = eventData.source;
        const entity = eventData.projectile;
        const hitEntityResult = eventData.getEntityHit();
        if (!hitEntityResult) return;
        const target = hitEntityResult.entity;
        if (!entity || !target) return;
        const identifier = entity.typeId;
        if (identifier !== Hedgehog.ENTITY_OR_ITEM_ID) return;
        target.applyDamage(2, { cause: EntityDamageCause.entityAttack, damagingEntity: player });
    }
    static spawnAndThrow(player, strength = 1, variant, isBaby, dye) {
        const headPos = player.getHeadLocation();
        const rot = player.getViewDirection();
        const hedgehog = player.dimension.spawnEntity(Hedgehog.ENTITY_OR_ITEM_ID, {
            x: headPos.x - rot.z * 0.3 + rot.x * 0.2,
            y: headPos.y - 0.2,  
            z: headPos.z + rot.x * 0.3 + rot.z * 0.2,
        });
        hedgehog.clearVelocity();
        hedgehog.applyImpulse({
            x: rot.x * strength + player.getVelocity().x,
            y: rot.y ,
            z: rot.z * strength + player.getVelocity().z,
        });
        Hedgehog.applyHedgehogProperties(hedgehog, variant, isBaby, dye, player);
        hedgehog.triggerEvent("sf_nba:projectile");
        hedgehog.setProperty("sf_nba:rolling", true);
        hedgehog.setDynamicProperty("sf_nba:last_thrown", Date.now());
        player.dimension.playSound("random.bow", player.location);
        system.runTimeout(() => hedgehog.triggerEvent('sf_nba:follow_owner'), 40)
        return hedgehog;
    }
    onLoad(e) {
        const { entity } = e
        if (entity.typeId != 'sf_nba:hedgehog') return
        const ownerId = entity.getDynamicProperty('sf_nba:owner_id')
        const owner = world.getPlayers().filter(p => p.id == ownerId)[0]
        const tameable = entity.getComponent('tameable')
        if (ownerId && !tameable.tamedToPlayerId && owner) {
            tameable.tame(owner)
        }
    }
    onPlayerLoad(e) {
        const { player } = e
        for (const dimension of dimensions) {
            const entities = world.getDimension(dimension).getEntities({ type: 'sf_nba:hedgehog' })
            for (const entity of entities) {
                const ownerId = entity.getDynamicProperty('sf_nba:owner_id')
                const tameable = entity.getComponent('tameable')
                if (ownerId == player.id && !tameable.tamedToPlayerId) {
                    tameable.tame(player)
                }
            }
        }
    }
}
new Hedgehog();